%
% FICHERO PARA DEMOSTRACION DE METODOS DE DIFERENCIAS FINITAS.
% ECUACION DE LAPLACE EN DOS DIMENSIONES CON DOMINIO CUADRADO.
%

%% 
% ejemplo de la ecuacion de Laplace con CC Direchlet por SOR
%
%          
%                           CN
%        |    ___________________________________ XNE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        | CW |                                  |CE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |__________________________________|
%        |   XSW              CS
%        |
%        |-------------------------------------------> columna
%        V
%      fila
%         
clear; clc;

% parametros fisicos

xsw = [  0.0  0.0];
xne = [ 10.0 10.0];

% parametros numericos

nx = [ 51 51];

% condiciones de contorno y estimacion inicial
u0 = zeros(nx);
u0n = 1.0;
u0w = 1.0;
u0s = 0.0;
u0e = 0.0;

u0( nx(1), :) = u0s; %Sur
u0( :, nx(2)) = u0e; %Este
u0( 1,:) = u0n;  % Norte
u0( :,1) = u0w;  % Oeste

% el laplaciano de cinco puntos no utiliza las esquinas
% les damos un valor 'ilusorio' como media entre los vecinos
u0(end, end) = 0.5*(u0(end-1,end)+u0(end,end-1));
u0(1,1)      = 0.5*(u0(2,1)      +u0(1,2)      );
u0(end,1)    = 0.5*(u0(end-1,1)  +u0(end,2)    );
u0(1,end)    = 0.5*(u0(2,end)    +u0(1,end-1)  );

alpha = 4.0/(2.0 + sqrt(4.0-(cos(pi/nx(1))+cos(pi/nx(2)))))-1.0;
%alpha = 1.1;
tolerance = 1.e-7;
maxiter = 1e4;
ncheck = 100;
[u niter] = laplacedirfdsor( xsw, xne, u0, alpha, tolerance, ...
                            maxiter, ncheck);

% realizamos representaciones graficas
nx = size(u);
dx = (xne-xsw)./(nx-1);
[ X1, X2] = meshgrid( xsw(1):dx(1):xne(1), ...
                      xne(2):(-dx(2)):xsw(2) );
                  
figure(1); clf;
surf( X1, X2, u);
view(-15, 30);

% comprobamos la bondad de la solucion
% calculamos laplaciano

fprintf('\nmaximo del laplaciano %e\n', max(max(del2(u(2:end-1,2:end-1)))));
fprintf('\nminimo del laplaciano %e\n', min(min(del2(u(2:end-1,2:end-1)))));

%

%%
% ecuacion de Laplace con condiciones de contorno Neumann por SOR
%
clear; clc;

% parametros fisicos

xsw = [  0.0  0.0];
xne = [ 10.0 10.0];

% parametros numericos

nx = [ 51 51];

% condiciones de contorno y estimacion inicial
u0 = ones(nx);
% derivadas normales (hacia el interior) en N S E W
cc = [ 1.0 2.0 0.0 -3.0];
%cc = [ 1.0 -1.0 0.0 0.0];
cc(1:2) = cc(1:2)/(xne(1)-xsw(1));
cc(3:4) = cc(3:4)/(xne(2)-xsw(2));

% alpha optimo segun la teoria
alpha = 4.0/(2.0 + sqrt(4.0-(cos(pi/nx(1))+cos(pi/nx(2)))))-1.0;
%alpha = 0.0;
tolerance = 1.e-5;
maxiter = 1e4;
ncheck = 20;
[u niter] = laplaceneufdsor( xsw, xne, u0, cc, alpha, tolerance, ...
                            maxiter, ncheck);

% comprobamos la bondad de la solucion
% calculamos laplaciano y representamos gradiente

fprintf('\nmaximo del laplaciano %e\n', max(max(del2(u))));
fprintf('\nminimo del laplaciano %e\n', min(min(del2(u))));

% realizamos representaciones graficas

nx = size(u);
dx = xne - xsw;
dx = dx./(nx-1);
[ X1, X2] = meshgrid( xsw(1):dx(1):xne(1), ...
                      xne(2):(-dx(2)):xsw(2) );

[u1, u2] = gradient( u, dx(1), (-dx(2)));
                  
figure(1); clf;                
surf( X1, X2, u);
view(26, 40);

figure(2);clf;
quiver( X1, X2, u1, u2); hold on; contour(X1, X2, u);

% dibujamos valores de las derivadas normales en contornos
% Estas derivadas normales NO SON LAS MISMAS QUE LAS EXIGIDAS,
% sino las evaluadas en puntos intermedios entre los nodos de la 
% malla.
figure(3);clf;
plot( u1(:,1)); hold on; plot(u1(:,end)); hold on; 
plot(u2(1,:)) ; hold on; plot(u2(end,:))

%